// ##############################################################
//  DataManager.m
//  Magic Number Machine
//
//  Created by Matt Gallagher on Sun Apr 20 2003.
//  Copyright (c) 2003 Matt Gallagher. All rights reserved.
// ##############################################################

#import "DataManager.h"
#import "InputManager.h"
#import "ExpressionDisplay.h"
#import "TreeHead.h"
#import "DrawerManager.h"
#import "ExpressionSymbols.h"
#import "Value.h"

//
// About the DataManager
//
// Magic Number Machine maintains a single window and with that window there is
// a single DataManager class that manages all the data for that window.
//
// This class contains the current expression (the data structure viewed through the
// main view of the main window) the history, data arrays and state variables.
//
// Little actual processing occurs in this class, it simply manages interaction between
// the input of data (input manager and sometimes the history or data arrays) and
// the current expression.
//

@implementation DataManager

// dennis: start ----------------------------------------------------------------------------------------------------
#pragma mark - Preference Handling

//
// Create a set of factory defaults
//
+ (void) initialize
{
	NSMutableDictionary *factoryDefaults = [NSMutableDictionary dictionary];
	
	[factoryDefaults setObject: [NSNumber numberWithInt:	10]		forKey: @"defaultRadix"];
	[factoryDefaults setObject: [NSNumber numberWithInt:	0 ]		forKey: @"defaultComplement"];
	
	[factoryDefaults setObject: [NSNumber numberWithInt:	0 ]		forKey: @"defaultDisplayType"];
	[factoryDefaults setObject: [NSNumber numberWithInt:	12]		forKey: @"defaultDigits"];
	[factoryDefaults setObject: [NSNumber numberWithInt:	3 ]		forKey: @"defaultSignificant"];
	[factoryDefaults setObject: [NSNumber numberWithInt:	3 ]		forKey: @"defaultFixed"];
	[factoryDefaults setObject: [NSNumber numberWithBool:	NO]		forKey: @"useThousandsSeparator"];

	[factoryDefaults setObject: [NSNumber numberWithInt: (int)BF_degrees]	forKey: @"defaultTrigMode"];
	
	[[NSUserDefaults standardUserDefaults] registerDefaults: factoryDefaults];
}

//
// Accessor-like methods for defaults
//
// Mainly for convenience.
//
- (void)setDefaultRadix:(int)base
{
	radix = base;
	[[NSUserDefaults standardUserDefaults] setInteger:radix forKey:@"defaultRadix"];
}

- (int)getDefaultRadixFromPref
{
	return [[NSUserDefaults standardUserDefaults] integerForKey:@"defaultRadix"];
}

- (void)setDefaultComplement:(int)bits
{
	complement = bits;
	[[NSUserDefaults standardUserDefaults] setInteger:bits forKey:@"defaultComplement"];
}

- (int)getDefaultComplementFromPref
{
	return [[NSUserDefaults standardUserDefaults] integerForKey:@"defaultComplement"];
}

- (void)setDefaultDisplayType:(int)displayType
{
	defaultDisplayType = displayType;
	[[NSUserDefaults standardUserDefaults] setInteger:displayType forKey:@"defaultDisplayType"];
}

- (int)getDefaultDisplayTypeFromPref
{
	return [[NSUserDefaults standardUserDefaults] integerForKey:@"defaultDisplayType"];
}

- (void)setDefaultDigits:(int)digits
{
	defaultDigits = MIN(digits, maximumLength);
	[[NSUserDefaults standardUserDefaults] setInteger:defaultDigits forKey:@"defaultDigits"];
}

- (int)getDefaultDigitsFromPref
{
	return [[NSUserDefaults standardUserDefaults] integerForKey:@"defaultDigits"];
}

- (void)setDefaultSignificant:(int)significant
{
	defaultSignificant = MIN(significant, maximumLength);
	[[NSUserDefaults standardUserDefaults] setInteger:defaultSignificant forKey:@"defaultSignificant"];
}

- (int)getDefaultSignificantFromPref
{
	return [[NSUserDefaults standardUserDefaults] integerForKey:@"defaultSignificant"];
}

- (void)setDefaultFixed:(int)fixed
{
	defaultFixed = MIN(fixed, 10);
	[[NSUserDefaults standardUserDefaults] setInteger:defaultFixed forKey:@"defaultFixed"];
}

- (int)getDefaultFixedFromPref
{
	return [[NSUserDefaults standardUserDefaults] integerForKey:@"defaultFixed"];
}

- (void)setDefaultThousandsSeparator:(BOOL)isUsed
{
	thousandsSeparator = isUsed;
	[[NSUserDefaults standardUserDefaults] setBool:isUsed forKey:@"useThousandsSeparator"];
}

- (BOOL)getDefaultThousandsSeparatorFromPref
{
	return [[NSUserDefaults standardUserDefaults] boolForKey:@"useThousandsSeparator"];
}

- (void)setDefaultTrigMode:(int)mode
{
	trigMode = (BFTrigMode)mode;
	[[NSUserDefaults standardUserDefaults] setInteger:mode forKey:@"defaultTrigMode"];
}

- (BFTrigMode)getDefaultTrigModeFromPref
{
	return (BFTrigMode)[[NSUserDefaults standardUserDefaults] integerForKey:@"defaultTrigMode"];
}

//
// dennis
//
// init (modified)
//
- (id)init
{
	self = [super init];
	if (self)
	{
		// Constants & states
		shiftIsDown           = NO;
		optionIsDown          = NO;
		shiftEnabledByToggle  = NO;
		optionEnabledByToggle = NO;
		equalsPressed         = NO;
		maximumLength         = 25;
		lengthLimitSave       = 0;
		fixedPlacesSave       = 0;
		fillLimitSave         = NO;
		
		// Defaults
		radix                 = [self getDefaultRadixFromPref];
		complement            = [self getDefaultComplementFromPref];
		thousandsSeparator    = [self getDefaultThousandsSeparatorFromPref];
		defaultDigits         = [self getDefaultDigitsFromPref];
		defaultSignificant    = [self getDefaultSignificantFromPref];
		defaultFixed          = [self getDefaultFixedFromPref];
		defaultDisplayType    = [self getDefaultDisplayTypeFromPref];
		trigMode              = [self getDefaultTrigModeFromPref];
		
		[self updateLengthLimit];
		
		currentExpression = nil;
		currentInputPoint = nil;
		
		// No expression and all empty data sets
		[TreeHead treeHeadWithValue:nil andManager:self];
		arrayDataArray = [[NSMutableArray arrayWithCapacity:0] retain];
		dataArray      = [[NSMutableArray arrayWithCapacity:0] retain];
		data2DArray    = [[NSMutableArray arrayWithCapacity:0] retain];
		historyArray   = [[NSMutableArray arrayWithCapacity:0] retain];
	}
	return self;
}

// dennis
//
// This is called when the preference is committed
//
- (void)saveDefaultsForThousands:(BOOL)separator 
						  digits:(int)digits 
					 significant:(int)significant 
						   fixed:(int)fixed 
						 display:(int)displayType
{
	[self setDefaultThousandsSeparator:separator];
	[self setDefaultDigits:digits];
	[self setDefaultSignificant:significant];
	[self setDefaultFixed:fixed];
	[self setDefaultDisplayType:displayType];
	
	[self updateExpressionDisplay];
}

- (void)updateLengthLimit
{
	switch (defaultDisplayType)
	{
		case 0:
			[self lengthLimit:defaultDigits fillLimit:NO fixedPlaces:0];
			break;
		case 1:
			[self lengthLimit:defaultSignificant fillLimit:YES fixedPlaces:0];
			break;
		case 2:
			[self lengthLimit:20 fillLimit:YES fixedPlaces:defaultFixed];
			break;
	}
}

- (void)updateRadixDisplay
{
	switch (radix)
	{
		case 2:
			[radixDisplay setStringValue:[[NSBundle bundleForClass:[self class]] localizedStringForKey:@"Radix: Binary" value:nil table:nil]];
			break;
		case 8:
			[radixDisplay setStringValue:[[NSBundle bundleForClass:[self class]] localizedStringForKey:@"Radix: Octal" value:nil table:nil]];
			break;
		case 10:
			[radixDisplay setStringValue:[[NSBundle bundleForClass:[self class]] localizedStringForKey:@"Radix: Decimal" value:nil table:nil]];
			break;
		case 16:
			[radixDisplay setStringValue:[[NSBundle bundleForClass:[self class]] localizedStringForKey:@"Radix: Hexadecimal" value:nil table:nil]];
			break;
	}
}

- (void)updatePrecisionDisplay
{
	if (complement != 0)
	{
		[precisionDisplay setStringValue:[NSString stringWithFormat:[[NSBundle bundleForClass:[self class]] localizedStringForKey:@"Precision: %d-bit 2's Complement" value:nil table:nil], complement]];
	}
	else
	{
		switch (defaultDisplayType)
		{
			case 0:
				[precisionDisplay setStringValue:[NSString stringWithFormat:[[NSBundle bundleForClass:[self class]] localizedStringForKey:@"Precision: %d digits" value:nil table:nil], defaultDigits]];
				break;
			case 1:
				[precisionDisplay setStringValue:[NSString stringWithFormat:[[NSBundle bundleForClass:[self class]] localizedStringForKey:@"Precision: %d significant figures" value:nil table:nil], defaultSignificant]];
				break;
			case 2:
				[precisionDisplay setStringValue:[NSString stringWithFormat:[[NSBundle bundleForClass:[self class]] localizedStringForKey:@"Precision: %d point places" value:nil table:nil], defaultFixed]];
				break;
		}
	}
}

- (void)updateExponentLeftShift
{
	if (fixedPlaces == 0 && complement == 0)
		[exponentLeftShift setEnabled:YES];
	else
		[exponentLeftShift setEnabled:NO];
}

- (void)updateTrigModeDisplay
{
	switch (trigMode)
	{
		case BF_degrees:
			[trigModeDisplay setStringValue:[[NSBundle bundleForClass:[self class]] localizedStringForKey:@"Trig. Mode: Degrees" value:nil table:nil]];
			break;
		case BF_radians:
			[trigModeDisplay setStringValue:[[NSBundle bundleForClass:[self class]] localizedStringForKey:@"Trig. Mode: Radians" value:nil table:nil]];
			break;
		case BF_gradians:
			[trigModeDisplay setStringValue:[[NSBundle bundleForClass:[self class]] localizedStringForKey:@"Trig. Mode: Gradians" value:nil table:nil]];
			break;
	}
}

- (void)updateExpressionDisplay
{
	[currentExpression refresh];
	[self valueChanged];
}


/* # dennis #
//
// init
//
// Constructor
//
- (id)init
{
	self = [super init];
	if (self)
	{
		// Initialise all the variables to the program defaults. A methodical programmer
		// would get these values from a file full of constants or better yet, save most of
		// them in the preference file. Just so you know: I don't care.
		radix = 10;
		shiftIsDown = NO;
		optionIsDown = NO;
		complement = 0;
		lengthLimitSave = 0;
		fixedPlacesSave = 0;
		fillLimitSave = NO;
		trigMode = BF_degrees;
		equalsPressed = NO;
		shiftEnabledByToggle = NO;
		optionEnabledByToggle = NO;
		maximumLength = 25;

		if ([[NSUserDefaults standardUserDefaults] stringForKey:@"defaultDigits"] == nil)
		{
			//	lengthLimit = 12;
			//	fillLimit = NO;
			//	fixedPlaces = 0;
			thousandsSeparator = NO;
			defaultDigits = 12;
			defaultSignificant = 3;
			defaultFixed = 3;
			defaultDisplayType = 0;
		}
		else
		{
			thousandsSeparator = [[NSUserDefaults standardUserDefaults] boolForKey:@"useThousandsSeparator"];
			defaultDigits = [[NSUserDefaults standardUserDefaults] integerForKey:@"defaultDigits"];
			defaultSignificant = [[NSUserDefaults standardUserDefaults] integerForKey:@"defaultSignificant"];
			defaultFixed = [[NSUserDefaults standardUserDefaults] integerForKey:@"defaultFixed"];
			defaultDisplayType = [[NSUserDefaults standardUserDefaults] integerForKey:@"defaultDisplayType"];
		}
		
		if ([[NSUserDefaults standardUserDefaults] stringForKey:@"defaultRadix"] != nil)
		{
			radix = [[NSUserDefaults standardUserDefaults] integerForKey:@"defaultRadix"];
			complement = [[NSUserDefaults standardUserDefaults] integerForKey:@"defaultComplement"];
		}
		
		if ([[NSUserDefaults standardUserDefaults] stringForKey:@"defaultTrigMode"] != nil)
		{
			trigMode = (BFTrigMode)[[NSUserDefaults standardUserDefaults] integerForKey:@"defaultTrigMode"];
		}
		
		switch (defaultDisplayType)
		{
			case 1:
				if (defaultSignificant > maximumLength)
					defaultSignificant = maximumLength;
				[self lengthLimit:defaultSignificant fillLimit:YES fixedPlaces:0];
				break;
			case 2:
				if (defaultFixed > 10)
					defaultFixed = 10;
				[self lengthLimit:20 fillLimit:YES fixedPlaces:defaultFixed];
				break;
			default:
			case 0:
				if (defaultDigits > maximumLength)
					defaultDigits = maximumLength;
				[self lengthLimit:defaultDigits fillLimit:NO fixedPlaces:0];
				break;
		}
		
		currentExpression = nil;
		currentInputPoint = nil;
		
		// No expression and all empty data sets
		[TreeHead treeHeadWithValue:nil andManager:self];
		arrayDataArray = [[NSMutableArray arrayWithCapacity:0] retain];
		dataArray = [[NSMutableArray arrayWithCapacity:0] retain];
		data2DArray = [[NSMutableArray arrayWithCapacity:0] retain];
		historyArray = [[NSMutableArray arrayWithCapacity:0] retain];
	}
	return self;
}
 */

//
// dealloc
//
// Destructor. Clean up the class and leave everything tidy
//
- (void)dealloc
{
	if (currentExpression != nil)
		[currentExpression release];

	[historyArray release];
	[arrayDataArray release];
	[dataArray release];
	[data2DArray release];
    [super dealloc];
}

//
// exportToPDF:
//
// Save the expression view to a PDF file.
//
- (IBAction)exportToPDF:(id)sender
{
	NSData *pdfData = [expressionDisplay pdfData];
	
	NSSavePanel *savePanel = [NSSavePanel savePanel];
	[savePanel setRequiredFileType:@"pdf"];

	//
	// Show the window as a modal sheet
	//
	[NSApp
		beginSheet:savePanel
		modalForWindow:[expressionDisplay window]
		modalDelegate:nil
		didEndSelector:nil
		contextInfo:pdfData];
	int result = [savePanel runModalForDirectory:NSHomeDirectory() file:nil];
	[NSApp endSheet:savePanel];
	
	if (result == NSFileHandlingPanelOKButton)
	{
		[pdfData writeToURL:[savePanel URL] atomically:YES];
	}
}

//
// addData
//
// Breaks the InputManager control of all the buttons by handling the "Add to Data" button.
// Appends the value in the expression display to the appropriate data array.
//
- (IBAction)addData:(id)sender
{
	BigCFloat	*numberCopy;
	
	if (!equalsPressed)
		[self equalsPressed];
	
	numberCopy = [[currentExpression getValue] copy];
	
	if (shiftIsDown && !optionIsDown)
	{
		[drawerManager addData2D:numberCopy];
	}
	else if (optionIsDown && !shiftIsDown)
	{
		[drawerManager addArrayData:numberCopy];
	}
	else
	{
		[drawerManager addData:numberCopy];
	}
}

//
// arrayData
//
// Allows access to the array data
//
- (NSMutableArray*)arrayData
{
	return arrayDataArray;
}

//
// awakeFromNib
//
// Fill out the display strings based on values restored from preferences
//
- (void)awakeFromNib
{
	/* # dennis #
	switch (radix)
	{
		case 2:
			[radixDisplay setStringValue:[[NSBundle bundleForClass:[self class]] localizedStringForKey:@"Radix: Binary" value:nil table:nil]];
			break;
		case 8:
			[radixDisplay setStringValue:[[NSBundle bundleForClass:[self class]] localizedStringForKey:@"Radix: Octal" value:nil table:nil]];
			break;
		case 10:
			[radixDisplay setStringValue:[[NSBundle bundleForClass:[self class]] localizedStringForKey:@"Radix: Decimal" value:nil table:nil]];
			break;
		case 16:
			[radixDisplay setStringValue:[[NSBundle bundleForClass:[self class]] localizedStringForKey:@"Radix: Hexadecimal" value:nil table:nil]];
			break;
	}

	switch (trigMode)
	{
		case BF_degrees:
			[trigModeDisplay setStringValue:[[NSBundle bundleForClass:[self class]] localizedStringForKey:@"Trig. Mode: Degrees" value:nil table:nil]];
			break;
		case BF_radians:
			[trigModeDisplay setStringValue:[[NSBundle bundleForClass:[self class]] localizedStringForKey:@"Trig. Mode: Radians" value:nil table:nil]];
			break;
		case BF_gradians:
			[trigModeDisplay setStringValue:[[NSBundle bundleForClass:[self class]] localizedStringForKey:@"Trig. Mode: Gradians" value:nil table:nil]];
			break;
	}
	 */
	[self updateRadixDisplay];		// dennis
	[self updatePrecisionDisplay];	// dennis
	[self updateTrigModeDisplay];	// dennis
}

//
// clearExpression
//
// Does what you'd expect... clears the whole expression.
//
- (void)clearExpression
{
	equalsPressed = NO;
	[TreeHead treeHeadWithValue:nil andManager:self];

	[self valueChanged];
}

//
// clearHistory
//
// Again, pretty obvious... clears the history.
//
- (IBAction)clearHistory:(id)sender
{
	[historyArray autorelease];
	historyArray = [[NSMutableArray arrayWithCapacity:0] retain];
	[drawerManager updateHistory];
}

//
// data
//
// Allows access to the data array.
//
- (NSMutableArray*)data
{
	return dataArray;
}

//
// data2D
//
// Allows access to the 2D data.
//
- (NSMutableArray*)data2D
{
	return data2DArray;
}

//
// ensureInputWithValue
//
// A behaviour thing... if the user presses equals, sometimes the next button pressed
// will use the result as the first number in the new expression, sometimes it will not.
// Calling this function with preserveValue set appropriately before 
//
- (void)ensureInputWithValue:(BOOL)preserveValue
{
	if (equalsPressed)
	{
		if (preserveValue)
		{
			[TreeHead treeHeadWithValue:[currentExpression getValue] andManager:self];
		}
		else
		{
			[TreeHead treeHeadWithValue:nil andManager:self];
		}
		
		equalsPressed = NO;
	}
}

//
// equalsPressed
//
// When equals is pressed we enable display of the result and append the expression to
// the history
//
- (void)equalsPressed
{
	if (equalsPressed)
		return;
	
	// Enable dispaly of the result
	equalsPressed = YES;
	[currentExpression equalsPressed];
	
	[self valueChanged];
	
	// Append the current expression to the history
	if ([currentExpression child] != nil)
	{
		[historyArray addObject:
			[NSArray arrayWithObjects:
				[NSKeyedArchiver archivedDataWithRootObject:[currentExpression child]],
				[expressionDisplay expressionPathFlipped],
				[NSNumber numberWithInt:[historyArray count] + 1],
				nil
			]
		];
		[drawerManager updateHistory];
	}
}

//
// getComplement
//
// Allows access to the complement variable
//
- (int)getComplement
{
	return complement;
}

//
// getCurrentExpression
//
// Allows access to the current expression
//
- (Expression*)getCurrentExpression
{
	return currentExpression;
}

//
// getCurrentExpression
//
// Returns whether or not the equals button has been pressed on the current expression
//
- (BOOL)getEqualsPressed
{
	return equalsPressed;
}

//
// getFillLimit
//
// Returns whether or not the equals button has been pressed on the current expression
//
- (BOOL)getFillLimit
{
	return fillLimit;
}

//
// getFixedPlaces
//
// Gives the number of fixes places in the display (if any)
//
- (unsigned int)getFixedPlaces
{
	return fixedPlaces;
}

//
// getInputPoint
//
// Gives a pointer to the current input point in the current expression
//
- (Expression*)getInputPoint
{
	return currentInputPoint;
}

//
// getLengthLimit
//
// Maximum number of digits. This returns it.
//
- (unsigned int)getLengthLimit
{
	return lengthLimit;
}

//
// getMaximumLength
//
// Maximum number of digits allowable at any time. This returns it.
//
- (unsigned int)getMaximumLength
{
	return maximumLength;
}

//
// getOption
//
// Returns whether the option key is down.
//
- (BOOL)getOption
{
	return optionIsDown;
}

//
// getRadix
//
// Returns the current radix.
//
- (short)getRadix
{
	return radix;
}

//
// getShift
//
// Tells whether the shift key is down
//
- (BOOL)getShift
{
	return shiftIsDown;
}

//
// getThousandsSeparator
//
// Returns whether to separate digits in groups.
//
- (BOOL)getThousandsSeparator
{
	return thousandsSeparator;
}

//
// getTrigMode
//
// Degrees, radians or gradians.
//
- (int)getTrigMode
{
	return trigMode;
}

//
// history
//
// Allows access to the history array.
//
- (NSMutableArray*)history
{
	return historyArray;
}

//
// lengthLimit
//
// Allows the "Disp" button sheets to set the precision, scientific notation or fixed
// display.
//
- (void)lengthLimit:(unsigned int)limit fillLimit:(BOOL)fill fixedPlaces:(unsigned int)places
{
	if (limit > maximumLength)
		limit = maximumLength;
	
	if (lengthLimitSave == 0)
	{
		lengthLimit = limit;
		fillLimit = fill;
		fixedPlaces = places;
	}
	else
	{
		lengthLimitSave = limit;
		fixedPlacesSave = places;
		fillLimitSave = fill;
		
		if (places > lengthLimit - 1)
			fixedPlaces = lengthLimit - 1;
		else
			fixedPlaces = places;
	}
	
	/* # dennis #
	 
	// Display the new precision
	if (fillLimit == NO)
		[precisionDisplay setStringValue:[NSString stringWithFormat:[[NSBundle bundleForClass:[self class]] localizedStringForKey:@"Precision: %d digits" value:nil table:nil], lengthLimit]];
	else if (fixedPlaces == 0)
		[precisionDisplay setStringValue:[NSString stringWithFormat:[[NSBundle bundleForClass:[self class]] localizedStringForKey:@"Precision: %d significant figures" value:nil table:nil], lengthLimit]];
	else
		[precisionDisplay setStringValue:[NSString stringWithFormat:[[NSBundle bundleForClass:[self class]] localizedStringForKey:@"Precision: %d point places" value:nil table:nil], lengthLimit]];
	
	// Refresh the entire display (radix change requires exponent reprocessign and digit
	// re-arranging
	[currentExpression refresh];
	[self valueChanged];
	
	// Set whether or not the left shift button is enabled.
	if (fixedPlaces == 0 && complement == 0)
		[exponentLeftShift setEnabled:YES];
	else
		[exponentLeftShift setEnabled:NO];
	 */
	
	[self updatePrecisionDisplay];	// dennis
	[self updateExponentLeftShift]; // dennis
	[self updateExpressionDisplay]; // dennis
}

//
// optionIsPressed
//
// Set the option key's state
//
- (void)optionIsPressed:(BOOL)isPressed
{
	if (optionIsDown != isPressed)
	{
		optionIsDown = isPressed;
		[optionButton highlight:optionIsDown];
	}
	
	optionEnabledByToggle = NO;
}

//
// optionToggled
//
// Toggle the option key's state
//
- (void)optionToggled
{
	optionIsDown = !optionIsDown;
	[optionButton highlight:optionIsDown];
	
	if (optionIsDown)
		optionEnabledByToggle = YES;
}

//
// setCurrentExpression
//
// Replace the current expression with the one that is passed in.
//
- (void)setCurrentExpression:(Expression*)newExpression
{
	if (currentExpression != nil)
		[currentExpression autorelease];
	
	currentExpression = [newExpression retain];
	[self valueChanged];
}

//
// setInputPoint
//
// Set the current input point in the current expression to the point passed in.
//
- (void)setInputPoint:(Expression*)point
{
	currentInputPoint = point;
}

//
// setInputAtPoint
//
// Determine the point clicked by the mouse and move the insertion point to it.
//
- (void)setInputAtPoint:(NSPoint)point
{
	Expression	*node;
	
	node = [currentExpression nodeContainingPoint:point];
	
	if (node == nil)
		node = currentExpression;

	equalsPressed = NO;
	[self setInputPoint:node];
	[self valueChanged];
}

//
// setRadix
//
// Respond to a radix change from the radix drawer
//
- (void)setRadix:(short)newRadix useComplement:(int)useComplement
{
	/* # dennis #
	radix = newRadix;
	complement = useComplement;
	
	[[NSUserDefaults standardUserDefaults] setInteger:radix forKey:@"defaultRadix"];
	[[NSUserDefaults standardUserDefaults] setInteger:useComplement forKey:@"defaultComplement"];
	*/
	[self setDefaultRadix: (int)newRadix];		// dennis
	[self setDefaultComplement: useComplement];	// dennis
	
	if (useComplement != 0)
	{
		if (lengthLimitSave == 0)
		{
			lengthLimitSave = lengthLimit;
			fixedPlacesSave = fixedPlaces;
			fillLimitSave = fillLimit;
		}
		lengthLimit = useComplement / (int)(log(radix) / log(2.0));
		
		fixedPlaces = 0;
		fillLimit = NO;
		
		/* dennis: the below is handled by updatePrecisionDisplay:
		[precisionDisplay setStringValue:[NSString stringWithFormat:[[NSBundle bundleForClass:[self class]] localizedStringForKey:@"Precision: %d-bit 2's Complement" value:nil table:nil], useComplement]];
		 */
	}
	else if (lengthLimitSave != 0)
	{
		lengthLimit = lengthLimitSave;
		fixedPlaces = fixedPlacesSave;
		fillLimit = fillLimitSave;
		lengthLimitSave = 0;
		fixedPlacesSave = 0;
		fillLimitSave = NO;

		/* dennis: the below is handled by updatePrecisionDisplay:
		if (fillLimit == NO)
			[precisionDisplay setStringValue:[NSString stringWithFormat:[[NSBundle bundleForClass:[self class]] localizedStringForKey:@"Precision: %d digits" value:nil table:nil], lengthLimit]];
		else if (fixedPlaces == 0)
			[precisionDisplay setStringValue:[NSString stringWithFormat:[[NSBundle bundleForClass:[self class]] localizedStringForKey:@"Precision: %d significant figures" value:nil table:nil], lengthLimit]];
		else
			[precisionDisplay setStringValue:[NSString stringWithFormat:[[NSBundle bundleForClass:[self class]] localizedStringForKey:@"Precision: %d point places" value:nil table:nil], lengthLimit]];
		 */
	}
	
	/* # dennis #
	switch (newRadix)
	{
	case 2:
		[radixDisplay setStringValue:[[NSBundle bundleForClass:[self class]] localizedStringForKey:@"Radix: Binary" value:nil table:nil]];
		break;
	case 8:
		[radixDisplay setStringValue:[[NSBundle bundleForClass:[self class]] localizedStringForKey:@"Radix: Octal" value:nil table:nil]];
		break;
	case 10:
		[radixDisplay setStringValue:[[NSBundle bundleForClass:[self class]] localizedStringForKey:@"Radix: Decimal" value:nil table:nil]];
		break;
	case 16:
		[radixDisplay setStringValue:[[NSBundle bundleForClass:[self class]] localizedStringForKey:@"Radix: Hexadecimal" value:nil table:nil]];
		break;
	}

	 [inputManager setControlsForRadix:radix];
	 [currentExpression refresh];
	 [self valueChanged];
	 
	 if (fixedPlaces == 0 && complement == 0)
	 [exponentLeftShift setEnabled:YES];
	 else
	 [exponentLeftShift setEnabled:NO];
	 
	 */
	
	[self updateRadixDisplay]; // dennis
	[self updatePrecisionDisplay];	// dennis
	[self updateExponentLeftShift]; // dennis
	[self updateExpressionDisplay]; // dennis
	
	[inputManager setControlsForRadix:radix]; // dennis
}

//
// setStartupState
//
// Tell all the other bits to go to first positions.
//
- (void)setStartupState
{
	[drawerManager setStartupState];
	[inputManager setControlsForRadix:radix];
	[inputManager setNextResponder:[self window]];
	// dennis // [inputManager setDefaultsForThousands:thousandsSeparator digits:defaultDigits significant:defaultSignificant fixed:defaultFixed display:(int)defaultDisplayType];
	
	// In 10.3, I started noticing a button update problem on startup. This is to try to fix that.
	[[self window] makeKeyAndOrderFront:self];
	[[self window] display];
	
	// Sanity check to ensure window is well-sized and visible (although this should be
	// enforced in the NIB file also).
	NSRect windowRect = [[self window] frame];
	NSRect screenRect = [[[self window] screen] frame];
	if (!NSIntersectsRect(windowRect, screenRect))
	{
		[[self window] setContentSize:NSMakeSize(330, 278)];
		[[self window] cascadeTopLeftFromPoint:NSZeroPoint];
	}
}

// dennis: setThousandsSeparator is no longer needed
//
// setThousandsSeparator
//
// Sets whether to separate digits in groups.
//
- (void)setThousandsSeparator:(BOOL)separator
{
	thousandsSeparator = separator;
}

//
// shiftIsPressed
//
// Set the current state of the shift button.
//
- (void)shiftIsPressed:(BOOL)isPressed
{
	if (shiftIsDown != isPressed)
	{
		shiftIsDown = isPressed;
		[shiftButton highlight:shiftIsDown];
	}
	
	shiftEnabledByToggle = NO;
}

//
// shiftResult
//
// Perform a shift left or a shift right on the current value.
//
- (void)shiftResult:(BOOL)left
{
	if (!equalsPressed)
		[self equalsPressed];
	
	[currentExpression shiftValue:(BOOL)left];
	[self valueChanged];
}

//
// shiftToggled
//
// Toggle the current state of the shift button.
//
- (void)shiftToggled
{
	shiftIsDown = !shiftIsDown;
	[shiftButton highlight:shiftIsDown];
	
	if (shiftIsDown)
		shiftEnabledByToggle = YES;
}

//
// shiftToggled
//
// Toggle the current state of the shift button.
//
- (void)trigModePressed
{
	if (!shiftIsDown && !optionIsDown)
	{
		trigMode = BF_degrees;
		// dennis: disabled // [trigModeDisplay setStringValue:[[NSBundle bundleForClass:[self class]] localizedStringForKey:@"Trig. Mode: Degrees" value:nil table:nil]];
	}
	else if (shiftIsDown)
	{
		trigMode = BF_radians;
		// dennis: disabled // [trigModeDisplay setStringValue:[[NSBundle bundleForClass:[self class]] localizedStringForKey:@"Trig. Mode: Radians" value:nil table:nil]];
	}
	else
	{
		trigMode = BF_gradians;
		// dennis: disabled // [trigModeDisplay setStringValue:[[NSBundle bundleForClass:[self class]] localizedStringForKey:@"Trig. Mode: Gradians" value:nil table:nil]];
	}
	
	// # dennis # // [[NSUserDefaults standardUserDefaults] setInteger:(int)trigMode forKey:@"defaultTrigMode"];
	[self setDefaultTrigMode:(int)trigMode];	// dennis
	[self updateTrigModeDisplay];				// dennis
}

//
// valueChanged
//
// This is called when something in the expression changes. Tells the expression
// display to update itself to reflect the change.
//
- (void)valueChanged
{
	[expressionDisplay expressionChanged];
	
	if (shiftEnabledByToggle)
		[self shiftIsPressed:NO];
	
	if (optionEnabledByToggle)
		[self optionIsPressed:NO];
}


//
// window
//
// Allows access to this object's window.
//
- (id)window
{
	return [expressionDisplay window];
}

- (BOOL)windowShouldClose:(id)sender
{
	[NSApp terminate:self];
	return YES;
}

@end
