// ##############################################################
//  DrawerManager.m
//  Magic Number Machine
//
//  Created by Matt Gallagher on Sun Apr 20 2003.
//  Copyright (c) 2003 Matt Gallagher. All rights reserved.
// ##############################################################

#import "DrawerManager.h"
#import "DataManager.h"
#import "DataFunctions.h"
#import "HistoryCell.h"
#import "Expression.h"

//
// About the DrawerManager
//
// The DrawerManager is an odd collection of both data, behaviour and input
// management for all of the drawers attached to the edge of the window. There
// is only one instance of the DrawerManager in the application.
//
// All the data which statically populates some tables in the drawers is statically
// defined in the consructor (ugly). This includes all the constants and the
// Data functions.
//
// The actual data kept in the "Data" drawers is owned by the DataManager class.
//

@implementation DrawerManager

//
// init
//
// Constructor. Performs the duty of filling the function tables and constants table. Not really
// something that should be loaded in a constructor but there you go.
//
- (id)init
{
	self = [super init];
	if (self)
	{
		activeDrawer = nil;
		numArrayColumns = 3;
		
		historyCell = [[[HistoryCell alloc] init] retain];
		
		arrayDataFunctionRows =
			[[NSArray arrayWithObjects:
				[NSArray arrayWithObjects:@"Gaussian Elimination", [NSValue value:&@selector(gaussianelimination:) withObjCType:@encode(SEL)], nil],
				[NSArray arrayWithObjects:@"Gaussian Elimination w/ Backsub", [NSValue value:&@selector(gaussianeliminationwithbacksub:) withObjCType:@encode(SEL)], nil],
				[NSArray arrayWithObjects:@"Determinant", [NSValue value:&@selector(determinant:) withObjCType:@encode(SEL)], nil],
				[NSArray arrayWithObjects:@"Inverse", [NSValue value:&@selector(inverse:) withObjCType:@encode(SEL)], nil],
				nil
			] retain];
		dataFunctionRows =
			[[NSArray arrayWithObjects:
				[NSArray arrayWithObjects:@"Sum", [NSValue value:&@selector(sum:) withObjCType:@encode(SEL)], nil],
				[NSArray arrayWithObjects:@"Mean (Average)", [NSValue value:&@selector(mean:) withObjCType:@encode(SEL)], nil],
				[NSArray arrayWithObjects:@"Mode (Most Frequent)", [NSValue value:&@selector(mode:) withObjCType:@encode(SEL)], nil],
				[NSArray arrayWithObjects:@"Median (Middle value)", [NSValue value:&@selector(median:) withObjCType:@encode(SEL)], nil],
				[NSArray arrayWithObjects:@"Variance", [NSValue value:&@selector(variance:) withObjCType:@encode(SEL)], nil],
				[NSArray arrayWithObjects:@"Standard Deviation", [NSValue value:&@selector(stddev:) withObjCType:@encode(SEL)], nil],
				[NSArray arrayWithObjects:@"Coefficient of Variation", [NSValue value:&@selector(coefofvariation:) withObjCType:@encode(SEL)], nil],
				nil
			] retain];
		data2DFunctionRows =
			[[NSArray arrayWithObjects:
				[NSArray arrayWithObjects:@"m (slope, Rank regression on y)", [NSValue value:&@selector(mfromrankregressionony:) withObjCType:@encode(SEL)], nil],
				[NSArray arrayWithObjects:@"b (y-intercept, Rank regression on y)", [NSValue value:&@selector(bfromrankregressionony:) withObjCType:@encode(SEL)], nil],
				[NSArray arrayWithObjects:@"a (x-intercept, Rank regression on y)", [NSValue value:&@selector(afromrankregressionony:) withObjCType:@encode(SEL)], nil],
				[NSArray arrayWithObjects:@"m (slope, Rank regression on x)", [NSValue value:&@selector(mfromrankregressiononx:) withObjCType:@encode(SEL)], nil],
				[NSArray arrayWithObjects:@"b (y-intercept, Rank regression on x)", [NSValue value:&@selector(bfromrankregressiononx:) withObjCType:@encode(SEL)], nil],
				[NSArray arrayWithObjects:@"a (x-intercept, Rank regression on x)", [NSValue value:&@selector(bfromrankregressiononx:) withObjCType:@encode(SEL)], nil],
				[NSArray arrayWithObjects:@"m (slope, regression on y, intercept at origin)", [NSValue value:&@selector(mfromrankregressiononywithoriginintercept:) withObjCType:@encode(SEL)], nil],
				[NSArray arrayWithObjects:@"m (slope, regression on x, intercept at origin)", [NSValue value:&@selector(mfromrankregressiononxwithoriginintercept:) withObjCType:@encode(SEL)], nil],
				nil
			] retain];
		
		radixDataRows = [[NSArray arrayWithObjects:
			[NSArray arrayWithObjects:[NSNumber numberWithInt:2], @"Binary", [NSNumber numberWithInt:0], nil],
			[NSArray arrayWithObjects:[NSNumber numberWithInt:8], @"Octal", [NSNumber numberWithInt:0], nil],
			[NSArray arrayWithObjects:[NSNumber numberWithInt:10], @"Decimal", [NSNumber numberWithInt:0], nil],
			[NSArray arrayWithObjects:[NSNumber numberWithInt:16], @"Hexadecimal", [NSNumber numberWithInt:0], nil],
			[NSArray arrayWithObjects:[NSNumber numberWithInt:2], @"8-bit Binary", [NSNumber numberWithInt:8], nil],
			[NSArray arrayWithObjects:[NSNumber numberWithInt:8], @"8-bit Octal", [NSNumber numberWithInt:8], nil],
			[NSArray arrayWithObjects:[NSNumber numberWithInt:16], @"8-bit Hex", [NSNumber numberWithInt:8], nil],
			[NSArray arrayWithObjects:[NSNumber numberWithInt:2], @"16-bit Binary", [NSNumber numberWithInt:16], nil],
			[NSArray arrayWithObjects:[NSNumber numberWithInt:8], @"16-bit Octal", [NSNumber numberWithInt:16], nil],
			[NSArray arrayWithObjects:[NSNumber numberWithInt:16], @"16-bit Hex", [NSNumber numberWithInt:16], nil],
			[NSArray arrayWithObjects:[NSNumber numberWithInt:2], @"32-bit Binary", [NSNumber numberWithInt:32], nil],
			[NSArray arrayWithObjects:[NSNumber numberWithInt:8], @"32-bit Octal", [NSNumber numberWithInt:32], nil],
			[NSArray arrayWithObjects:[NSNumber numberWithInt:16], @"32-bit Hex", [NSNumber numberWithInt:32], nil],
			[NSArray arrayWithObjects:[NSNumber numberWithInt:2], @"64-bit Binary", [NSNumber numberWithInt:64], nil],
			[NSArray arrayWithObjects:[NSNumber numberWithInt:8], @"64-bit Octal", [NSNumber numberWithInt:64], nil],
			[NSArray arrayWithObjects:[NSNumber numberWithInt:16], @"64-bit Hex", [NSNumber numberWithInt:64], nil],
			nil
		] retain];
		
		constantsDataRows =
			[[NSArray arrayWithObjects:
				[NSArray arrayWithObjects:[NSString stringWithUTF8String:"a_0	Bohr radius (m)"], [BigCFloat bigFloatWithDouble:5.29177249e-11 radix:10], nil],
				[NSArray arrayWithObjects:[NSString stringWithUTF8String:"α	Fine structure constant"], [BigCFloat bigFloatWithDouble:0.00729735308 radix:10], nil],
				[NSArray arrayWithObjects:[NSString stringWithUTF8String:"atm	Standard atmosphere (Pa)"], [BigCFloat bigFloatWithDouble:101325 radix:10], nil],
				[NSArray arrayWithObjects:[NSString stringWithUTF8String:"b	Wien displacement law constant (m K)"], [BigCFloat bigFloatWithDouble:0.002897756 radix:10], nil],
				[NSArray arrayWithObjects:[NSString stringWithUTF8String:"c_1	First radiation constant (W m2)"], [BigCFloat bigFloatWithDouble:3.7417749e-16 radix:10], nil],
				[NSArray arrayWithObjects:[NSString stringWithUTF8String:"c_2	Second radiation constant (m K)"], [BigCFloat bigFloatWithDouble:0.01438769 radix:10], nil],
				[NSArray arrayWithObjects:[NSString stringWithUTF8String:"c	Speed of light in vacuum (m s-1)"], [BigCFloat bigFloatWithDouble:299792458 radix:10], nil],
				[NSArray arrayWithObjects:[NSString stringWithUTF8String:"E_h	Hartree energy (J)"], [BigCFloat bigFloatWithDouble:4.3597482e-18 radix:10], nil],
				[NSArray arrayWithObjects:[NSString stringWithUTF8String:"e	Elementary charge (C)"], [BigCFloat bigFloatWithDouble:1.60217733e-19 radix:10], nil],
				[NSArray arrayWithObjects:[NSString stringWithUTF8String:"ε_0	Permittivity of vacuum (F m-1)"], [BigCFloat bigFloatWithDouble:8.85E-012 radix:10], nil],
				[NSArray arrayWithObjects:[NSString stringWithUTF8String:"eV	Electron volt (J)"], [BigCFloat bigFloatWithDouble:1.60217733e-19 radix:10], nil],
				[NSArray arrayWithObjects:[NSString stringWithUTF8String:"F	Faraday constant (C mol-1)"], [BigCFloat bigFloatWithDouble:96485.309 radix:10], nil],
				[NSArray arrayWithObjects:[NSString stringWithUTF8String:"g_e	Electron g-factor"], [BigCFloat bigFloatWithDouble:2.002319304386 radix:10], nil],
				[NSArray arrayWithObjects:[NSString stringWithUTF8String:"g_µ	Muon g-factor"], [BigCFloat bigFloatWithDouble:2.002331846 radix:10], nil],
				[NSArray arrayWithObjects:[NSString stringWithUTF8String:"g_n	Standard acceleration of gravity (m s-2)"], [BigCFloat bigFloatWithDouble:9.81 radix:10], nil],
				[NSArray arrayWithObjects:[NSString stringWithUTF8String:"G	Gravitational constant (m3 kg-1 s-2)"], [BigCFloat bigFloatWithDouble:6.67E-011 radix:10], nil],
				[NSArray arrayWithObjects:[NSString stringWithUTF8String:"h	Planck constant (J s)"], [BigCFloat bigFloatWithDouble:6.63E-034 radix:10], nil],
				[NSArray arrayWithObjects:[NSString stringWithUTF8String:"h	h-bar (J s)"], [BigCFloat bigFloatWithDouble:1.05457266e-34 radix:10], nil],
				[NSArray arrayWithObjects:[NSString stringWithUTF8String:"i	square-root of -1"], [BigCFloat bigFloatWithReal:[BigFloat bigFloatWithInt:0 radix:10] imaginary:[BigFloat bigFloatWithInt:1 radix:10]], nil],
				[NSArray arrayWithObjects:[NSString stringWithUTF8String:"k	Boltzmann constant (J K-1)"], [BigCFloat bigFloatWithDouble:1.380658e-23 radix:10], nil],
				[NSArray arrayWithObjects:[NSString stringWithUTF8String:"l_p	Planck length (m)"], [BigCFloat bigFloatWithDouble:1.61605e-35 radix:10], nil],
				[NSArray arrayWithObjects:[NSString stringWithUTF8String:"λ_C	Electron Compton wavelength/2π (m)"], [BigCFloat bigFloatWithDouble:3.86159323e-13 radix:10], nil],
				[NSArray arrayWithObjects:[NSString stringWithUTF8String:"λ_C,n	Neutron Compton wavelength (m)"], [BigCFloat bigFloatWithDouble:1.3195911e-15 radix:10], nil],
				[NSArray arrayWithObjects:[NSString stringWithUTF8String:"λ_C,p	Proton Compton wavelength (m)"], [BigCFloat bigFloatWithDouble:1.32141002e-15 radix:10], nil],
				[NSArray arrayWithObjects:[NSString stringWithUTF8String:"λ_C	Electron Compton wavelength (m)"], [BigCFloat bigFloatWithDouble:2.42631058e-12 radix:10], nil],
				[NSArray arrayWithObjects:[NSString stringWithUTF8String:"m_d	Deuteron mass (kg)"], [BigCFloat bigFloatWithDouble:3.343586e-27 radix:10], nil],
				[NSArray arrayWithObjects:[NSString stringWithUTF8String:"m_e	Electron mass (Kg)"], [BigCFloat bigFloatWithDouble:9.1093897e-31 radix:10], nil],
				[NSArray arrayWithObjects:[NSString stringWithUTF8String:"m_n	Neutron mass (kg)"], [BigCFloat bigFloatWithDouble:1.6749286e-27 radix:10], nil],
				[NSArray arrayWithObjects:[NSString stringWithUTF8String:"m_p	Planck mass (kg)"], [BigCFloat bigFloatWithDouble:2.17671e-08 radix:10], nil],
				[NSArray arrayWithObjects:[NSString stringWithUTF8String:"m_p	Proton mass (kg)"], [BigCFloat bigFloatWithDouble:1.6726231e-27 radix:10], nil],
				[NSArray arrayWithObjects:[NSString stringWithUTF8String:"m_u	Atomic mass constant (kg)"], [BigCFloat bigFloatWithDouble:1.6605402e-27 radix:10], nil],
				[NSArray arrayWithObjects:[NSString stringWithUTF8String:"µ_0	Permeability of vacuum (N A-2)"], [BigCFloat bigFloatWithDouble:1.26E-006 radix:10], nil],
				[NSArray arrayWithObjects:[NSString stringWithUTF8String:"µ_B	Bohr magneton (J T-1)"], [BigCFloat bigFloatWithDouble:9.2740154e-24 radix:10], nil],
				[NSArray arrayWithObjects:[NSString stringWithUTF8String:"µ_d	Deuteron magnetic moment (J T-1)"], [BigCFloat bigFloatWithDouble:4.3307375e-27 radix:10], nil],
				[NSArray arrayWithObjects:[NSString stringWithUTF8String:"µ_N	Nuclear magneton (J T-1)"], [BigCFloat bigFloatWithDouble:5.0507866e-27 radix:10], nil],
				[NSArray arrayWithObjects:[NSString stringWithUTF8String:"n_0	Loschmidt constant (m-3)"], [BigCFloat bigFloatWithDouble:2.686763e+25 radix:10], nil],
				[NSArray arrayWithObjects:[NSString stringWithUTF8String:"N_A	Avagadro constant (mol-1)"], [BigCFloat bigFloatWithDouble:6.0221367e+23 radix:10], nil],
				[NSArray arrayWithObjects:[NSString stringWithUTF8String:"φ_0	Magnetic flux quantum (Wb)"], [BigCFloat bigFloatWithDouble:2.06783461e-15 radix:10], nil],
				[NSArray arrayWithObjects:[NSString stringWithUTF8String:"π	Pi"], [BigCFloat bigFloatWithDouble:3.14159265358979323846264338 radix:10], nil],
				[NSArray arrayWithObjects:[NSString stringWithUTF8String:"r_e	Electron classical radius (m)"], [BigCFloat bigFloatWithDouble:2.81794092e-15 radix:10], nil],
				[NSArray arrayWithObjects:[NSString stringWithUTF8String:"R_H	Quantized Hall resistance (Ω)"], [BigCFloat bigFloatWithDouble:25812.8056 radix:10], nil],
				[NSArray arrayWithObjects:[NSString stringWithUTF8String:"R	Molar gas constant (J mol-1 K-1)"], [BigCFloat bigFloatWithDouble:8.31451 radix:10], nil],
				[NSArray arrayWithObjects:[NSString stringWithUTF8String:"Ry	Rydberg constant (m-1)"], [BigCFloat bigFloatWithDouble:10973731.534 radix:10], nil],
				[NSArray arrayWithObjects:[NSString stringWithUTF8String:"σ_e	Electron Thomson cross section (m2)"], [BigCFloat bigFloatWithDouble:6.6524616e-29 radix:10], nil],
				[NSArray arrayWithObjects:[NSString stringWithUTF8String:"σ	Stefan-Boltzmann const. (W m-2 K-4)"], [BigCFloat bigFloatWithDouble:5.67051e-08 radix:10], nil],
				[NSArray arrayWithObjects:[NSString stringWithUTF8String:"t_p	Planck time (s)"], [BigCFloat bigFloatWithDouble:5.39056e-44 radix:10], nil],
				[NSArray arrayWithObjects:[NSString stringWithUTF8String:"u	Atomic mass unit (kg)"], [BigCFloat bigFloatWithDouble:1.6605402e-27 radix:10], nil],
				[NSArray arrayWithObjects:[NSString stringWithUTF8String:"V_m	Molar vol. (ideal gas at STP) (m3 mol-1)"], [BigCFloat bigFloatWithDouble:0.0224141 radix:10], nil],
				nil
			] retain];
	}
	return self;
}

//
// dealloc
//
// Cleans everything up (I hope)
//
- (void)dealloc
{
	if (activeDrawer)
		[activeDrawer release];
	
	[radixDataRows release];
	[historyCell release];
	[constantsDataRows release];
	[arrayDataFunctionRows release];
	[dataFunctionRows release];
	[data2DFunctionRows release];
    [super dealloc];
}

//
// addArrayData
//
// Adds a value to the array data, updating the table to display the change
//
- (void)addArrayData:(BigCFloat*)value
{
	[[dataManager arrayData] addObject:value];
	[self updateArrayDataArray];
	return;
}

//
// addData
//
// Adds a value to the data, updating the table to display the change
//
- (void)addData:(BigCFloat*)value
{
	if ([dataTableView selectedRow] < 0 ||
		[dataTableView selectedRow] > [[dataManager data] count])
	{
		[[dataManager data] addObject:value];
		[self updateDataArray];
		return;
	}
	
	[[dataManager data] replaceObjectAtIndex:[dataTableView selectedRow] withObject:value];
	[self updateDataArray];
}

//
// addData2D
//
// Adds a value to the 2D data, updating the table to display the change
//
- (void)addData2D:(BigCFloat*)value
{
	[[dataManager data2D] addObject:value];
	[self updateData2DArray];
	return;
}

//
// arrayColumnsChanged
//
// Changes the number of columns in the array data.
//
- (IBAction)arrayColumnsChanged:(id)sender
{
	NSTableColumn	*column;

	numArrayColumns = [sender intValue];
	
	while ([arrayDataTableView numberOfColumns] > numArrayColumns)
	{
		int numberOfColumns = [arrayDataTableView numberOfColumns];
		column = [arrayDataTableView tableColumnWithIdentifier:[NSString stringWithFormat:@"%d", numberOfColumns - 1]];
		[arrayDataTableView removeTableColumn:column];
	}
	while ([arrayDataTableView numberOfColumns] < numArrayColumns)
	{
		column = [[[NSTableColumn alloc] initWithIdentifier:[NSString stringWithFormat:@"%d", [arrayDataTableView numberOfColumns]]] autorelease];
		[[column headerCell] setStringValue:[NSString stringWithFormat:@"%d", [arrayDataTableView numberOfColumns] + 1]];
		[column setWidth:60.0];
		[arrayDataTableView addTableColumn:column];
	}
	
	[self updateArrayDataArray];
}

//
// applyArrayDataFunction
//
// Calls the currently selected function and applies it to the array data
//
- (IBAction)applyArrayDataFunction:(id)sender
{
	BigCFloat *result;
	SEL		method;
	
	if ([[dataManager arrayData] count] == 0)
	{
		return;
	}
	
	[[[arrayDataFunctionRows objectAtIndex:[arrayDataFunctionsTableView selectedRow]] objectAtIndex:1] getValue:&method];
	
	result = [dataFunctions performSelector:method withObject:[dataManager arrayData]	];

	if (result != nil)
	{
		[dataManager ensureInputWithValue:NO];
		[[dataManager getInputPoint] valueInserted:result];
		[dataManager valueChanged];
	}
	else
	{
		[self updateArrayDataArray];
	}
}

//
// applyData2DFunction
//
// Calls the currently selected function and applies it to the 2D data
//
- (IBAction)applyData2DFunction:(id)sender
{
	BigCFloat *result;
	SEL		method;
	
	if ([[dataManager data2D] count] == 0)
	{
		return;
	}
	
	[[[data2DFunctionRows objectAtIndex:[data2DFunctionsTableView selectedRow]] objectAtIndex:1] getValue:&method];
	
	result = [dataFunctions performSelector:method withObject:[dataManager data2D]];
	
	if (result != nil)
	{
		[dataManager ensureInputWithValue:NO];
		[[dataManager getInputPoint] valueInserted:result];
		[dataManager valueChanged];
	}
}

//
// applyDataFunction
//
// Calls the currently selected function and applies it to the data values
//
- (IBAction)applyDataFunction:(id)sender
{
	BigCFloat *result;
	SEL		method;
	
	if ([[dataManager data] count] == 0)
	{
		return;
	}
	
	[[[dataFunctionRows objectAtIndex:[dataFunctionsTableView selectedRow]] objectAtIndex:1] getValue:&method];
	
	result = [dataFunctions performSelector:method withObject:[dataManager data]];
	
	if (result != nil)
	{
		[dataManager ensureInputWithValue:NO];
		[[dataManager getInputPoint] valueInserted:result];
		[dataManager valueChanged];
	}
}

//
// changeRadix
//
// Dispatches a change radix instruction to the data manager after the user clicks on a row
// in the radix table.
//
- (IBAction)changeRadix:(id)sender
{
	[dataManager setRadix:[[[radixDataRows objectAtIndex:[sender clickedRow]] objectAtIndex:0] intValue] useComplement:[[[radixDataRows objectAtIndex:[sender clickedRow]] objectAtIndex:2] intValue]];
}

//
// clearAllArrayDataValues
//
// Empties the array data
//
- (IBAction)clearAllArrayDataValues:(id)sender
{
	[[dataManager arrayData] removeAllObjects];
	[self updateArrayDataArray];
}

//
// clearAllData2DValues
//
// Empties the 2D data
//
- (IBAction)clearAllData2DValues:(id)sender
{
	[[dataManager data2D] removeAllObjects];
	[self updateData2DArray];
}

//
// clearAllDataValues
//
// Empties the data
//
- (IBAction)clearAllDataValues:(id)sender
{
	[[dataManager data] removeAllObjects];
	[self updateDataArray];
}

//
// clearArrayValue
//
// Clears the currently selected row in the array data
//
- (IBAction)clearArrayValue:(id)sender
{
	int i;
	
	if ([arrayDataTableView selectedRow] == -1)
	{
		return;
	}
	
	for (i = 0; i < numArrayColumns; i++)
	{
		if ([arrayDataTableView selectedRow] * numArrayColumns < [[dataManager arrayData] count])
		{
			[[dataManager arrayData] removeObjectAtIndex:[arrayDataTableView selectedRow] * numArrayColumns];
		}
	}
	[self updateArrayDataArray];
}

//
// clearData2DValue
//
// Clears the currently selected row in the 2D data
//
- (IBAction)clearData2DValue:(id)sender
{
	if ([data2DTableView selectedRow] == -1)
	{
		return;
	}
	
	[[dataManager data2D] removeObjectAtIndex:[data2DTableView selectedRow] * 2];
	if ([data2DTableView selectedRow] * 2 < [[dataManager data2D] count])
		[[dataManager data2D] removeObjectAtIndex:[data2DTableView selectedRow] * 2];
	[self updateData2DArray];
}

//
// clearDataValue
//
// Clears the currently selected row in the data
//
- (IBAction)clearDataValue:(id)sender
{
	[[dataManager data] removeObjectAtIndex:[dataTableView selectedRow]];
	[self updateDataArray];
}

//
// constantSelected
//
// Inserts the currently selected constant in the current expression
//
- (IBAction)constantSelected:(id)sender;
{
	BigCFloat		*pasteValue;
	Expression	*inputPoint;
	
	if
	(
		[sender clickedRow] == -1
		||
		[[constantsDataRows objectAtIndex:[sender clickedRow]] count] == 0
	)
		return;
	
	pasteValue = [[constantsDataRows objectAtIndex:[sender clickedRow]] objectAtIndex:1];
	[dataManager ensureInputWithValue:NO];
	inputPoint = [dataManager getInputPoint];
	[inputPoint valueInserted:pasteValue];
	[dataManager valueChanged];
}

//
// copyDataValueToDisplay
//
// Inserts the value currently selected in the data table into the current expression
//
- (IBAction)copyDataValueToDisplay:(id)sender
{
	if ([dataTableView selectedRow] == -1)
		return;
	
	[dataManager ensureInputWithValue:NO];
	[[dataManager getInputPoint] valueInserted:[[dataManager data] objectAtIndex:[dataTableView selectedRow]]];
	[dataManager valueChanged];
}

//
// historySelected
//
// Inserts the currently selected expression in the history into the current expression
//
- (void)historySelected:(id)sender
{
	Expression	*pasteExpression;
	Expression	*inputPoint;
	
	if ([sender clickedRow] == -1)
		return;
	
	pasteExpression = [NSKeyedUnarchiver unarchiveObjectWithData:[[[dataManager history] objectAtIndex:[sender clickedRow]] objectAtIndex:0]];
	[dataManager ensureInputWithValue:NO];
	inputPoint = [dataManager getInputPoint];
	[inputPoint bracketPressed];
	inputPoint = [dataManager getInputPoint];
	[inputPoint expressionInserted:pasteExpression];
	inputPoint = [dataManager getInputPoint];
	[inputPoint closeBracketPressed];
	[dataManager valueChanged];
}

//
// numberOfArrayColumns
//
// Returns the current number of columns used in the array data.
//
- (int)numberOfArrayColumns
{
	return numArrayColumns;
}

//
// numberOfRowsInTableView
//
// Tells the different tables how many rows of data there are in their respective data sets.
//
- (int)numberOfRowsInTableView:(NSTableView *)aTableView
{
	if ([aTableView isEqualTo:arrayDataTableView])
	{
		return ([[dataManager arrayData] count] + (numArrayColumns - 1)) / numArrayColumns;
	}
	else if ([aTableView isEqualTo:arrayDataFunctionsTableView])
	{
		return [arrayDataFunctionRows count];
	}
	else if ([aTableView isEqualTo:constantsTableView])
	{
		return [constantsDataRows count];
	}
	else if ([aTableView isEqualTo:dataTableView])
	{
		return [[dataManager data] count];
	}
	else if ([aTableView isEqualTo:dataFunctionsTableView])
	{
		return [dataFunctionRows count];
	}
	else if ([aTableView isEqualTo:data2DTableView])
	{
		return ([[dataManager data2D] count] + 1) / 2;
	}
	else if ([aTableView isEqualTo:data2DFunctionsTableView])
	{
		return [data2DFunctionRows count];
	}
	else if ([aTableView isEqualTo:historyTableView])
	{
		return [[dataManager history] count];
	}
	else if ([aTableView isEqualTo:radixTableView])
	{
		return [radixDataRows count];
	}
	else
	{
		return 0;
	}
}

//
// setStartupState
//
// At startup, some of the tables need to be redrawn (bug?) and select the current radix and
// set the class for drawing the history table.
//
- (void)setStartupState
{
	int offset = 0;
	int cluster = 0;
	
	// dennis: selecting the right row is trickier as base 10 is not available for complement other than 0.
	// perhaps a for loop is easier. nevertheless i'll 'patch' over the existing code.
	
	switch ([dataManager getRadix])
	{
		case 2:
			offset = 0;
			break;
		case 8:
			offset = 1;
			break;
		case 10:
			offset = 2;
			break;
		case 16:
			offset = 3;
			break;
	}
	switch ([dataManager getComplement])
	{
		case 0:
			cluster = 0;
			break;
		case 8:
			cluster = 1;
			break;
		case 16:
			cluster = 2;
			break;
		case 32:
			cluster = 3;
			break;
		case 64:
			cluster = 4;
			break;
	}
	
	// dennis: disabled // [radixTableView selectRow:cluster * 4 + offset byExtendingSelection:NO];
	
	// dennis: start -----------------------
	
	// For complement > 0, we now have 3 radices and the offset is different too.
		
	if ([dataManager getComplement] > 0)
	{
		switch ([dataManager getRadix])
		{
			case 2:
				offset = 1;
				break;
			case 8:
				offset = 2;
				break;
			case 16:
				offset = 3;
				break;
		}
	}
	[radixTableView selectRow:cluster * 3 + offset byExtendingSelection:NO]; // dennis
	[dataManager setRadix:[dataManager getRadix] useComplement:[dataManager getComplement]]; // dennis: ensures the format is displayed properly

	
	// dennis: end -----------------------

	
	[[[historyTableView tableColumns] objectAtIndex:0] setDataCell:historyCell];
	
	[data2DFunctionsTableView reloadData];
	[arrayDataFunctionsTableView reloadData];
	[self updateHistory];
	
	[[arrayDataDrawer contentView] setNextResponder:[dataManager window]];
	[[data2DDrawer contentView] setNextResponder:[dataManager window]];
	[[dataDrawer contentView] setNextResponder:[dataManager window]];
	[[historyDrawer contentView] setNextResponder:[dataManager window]];
	[[radixDrawer contentView] setNextResponder:[dataManager window]];
	[[constantsDrawer contentView] setNextResponder:[dataManager window]];
}

//
// tableView
//
// Tells the table views what object is at each table location so that it can draw itself.
//
- (id)tableView:(NSTableView*)aTableView objectValueForTableColumn:(NSTableColumn*)aTableColumn row:(int)rowIndex
{
	if ([aTableView isEqualTo:arrayDataTableView])
	{
		int count = [[dataManager arrayData] count];
		int column = [[aTableColumn identifier] intValue];
		BigCFloat *object;
		
		if (rowIndex * numArrayColumns + [[aTableColumn identifier] intValue] >= count)
			return @"";
		
		object = [[dataManager arrayData] objectAtIndex:rowIndex * numArrayColumns + column];
		
		return [object toShortString:3];
	}
	else if ([aTableView isEqualTo:arrayDataFunctionsTableView])
	{
		return [[arrayDataFunctionRows objectAtIndex:rowIndex] objectAtIndex:0];
	}
	else if ([aTableView isEqualTo:constantsTableView])
	{
		if ([[constantsDataRows objectAtIndex:rowIndex] count] != 0)
			return [[constantsDataRows objectAtIndex:rowIndex] objectAtIndex:0];
		else
			return @"";
	}
	else if ([aTableView isEqualTo:dataTableView])
	{
		return [[[dataManager data] objectAtIndex:rowIndex] toShortString:7];
	}
	else if ([aTableView isEqualTo:dataFunctionsTableView])
	{
		return [[dataFunctionRows objectAtIndex:rowIndex] objectAtIndex:0];
	}
	else if ([aTableView isEqualTo:data2DTableView])
	{
		if (rowIndex * 2 + [[aTableColumn identifier] intValue] >= [[dataManager data2D] count])
			return @"";
		
		return [[[dataManager data2D] objectAtIndex:rowIndex * 2 + [[aTableColumn identifier] intValue]] toShortString:3];
	}
	else if ([aTableView isEqualTo:data2DFunctionsTableView])
	{
		return [[data2DFunctionRows objectAtIndex:rowIndex] objectAtIndex:0];
	}
	else if ([aTableView isEqualTo:historyTableView])
	{
		return [[dataManager history] objectAtIndex:rowIndex];
	}
	else if ([aTableView isEqualTo:radixTableView])
	{
		return [[radixDataRows objectAtIndex:rowIndex] objectAtIndex:[[aTableColumn identifier] intValue]];
	}
	else
	{
		return 0;
	}
}

//
// toggleDrawer
//
// When the user presses a drawer button, we need to retract any open drawers and open
// the selected one (or toggle the selected one if it is already open).
//
- (IBAction)toggleDrawer:(id)sender
{
	BOOL	close = NO;
	
	if (activeDrawer != nil)
	{
		[activeDrawer close];
	}
	
	switch([sender tag])
	{
	case 0:
		if (activeDrawer != nil && [activeDrawer isEqualTo:historyDrawer])
			close = YES;
		else
			activeDrawer = historyDrawer;
		break;
	case 1:
		if (activeDrawer != nil && [activeDrawer isEqualTo:radixDrawer])
			close = YES;
		else
			activeDrawer = radixDrawer;
		break;
	case 2:
		if (activeDrawer != nil && [activeDrawer isEqualTo:dataDrawer])
			close = YES;
		else
			activeDrawer = dataDrawer;
		break;
	case 3:
		if (activeDrawer != nil && [activeDrawer isEqualTo:data2DDrawer])
			close = YES;
		else
			activeDrawer = data2DDrawer;
		break;
	case 4:
		if (activeDrawer != nil && [activeDrawer isEqualTo:arrayDataDrawer])
			close = YES;
		else
			activeDrawer = arrayDataDrawer;
		break;
	case 5:
		if (activeDrawer != nil && [activeDrawer isEqualTo:constantsDrawer])
			close = YES;
		else
			activeDrawer = constantsDrawer;
		break;
	}
	
	if (close)
	{
		if ([activeDrawer state] != NSDrawerClosingState)
			[activeDrawer openOnEdge:NSMinXEdge];
		else
			activeDrawer = nil;
	}
	else
	{
		[activeDrawer setMinContentSize:NSMakeSize(260, 200)];
		[activeDrawer openOnEdge:NSMinXEdge];
	}
}

//
// updateArrayDataArray
//
// Refresh and rescroll the array data table.
//
- (void)updateArrayDataArray
{
	[arrayDataTableView reloadData];
	
	if
	(
		[arrayDataTableView selectedColumn] == -1
		||
		[arrayDataTableView selectedRow] == -1
	)
	{
		[arrayDataTableView scrollRowToVisible:([[dataManager arrayData] count] - 1) / numArrayColumns];
	}
	else
	{
		[arrayDataTableView scrollRowToVisible:[arrayDataTableView selectedRow]];
		[arrayDataTableView scrollColumnToVisible:[arrayDataTableView selectedColumn]];
	}
}

//
// updateData2DArray
//
// Refresh and rescroll the 2D data table.
//
- (void)updateData2DArray
{
	[data2DTableView reloadData];
	
	if
	(
		[data2DTableView selectedColumn] == -1
		||
		[data2DTableView selectedRow] == -1
	)
	{
		[data2DTableView scrollRowToVisible:([[dataManager data2D] count] - 1) / 2];
	}
	else
	{
		[data2DTableView scrollRowToVisible:[data2DTableView selectedRow]];
		[data2DTableView scrollColumnToVisible:[data2DTableView selectedColumn]];
	}
}

//
// updateDataArray
//
// Refresh and rescroll the data table.
//
- (void)updateDataArray
{
	[dataTableView reloadData];
	
	if ([dataTableView selectedRow] == -1)
	{
		[dataTableView scrollRowToVisible:[[dataManager data] count]];
	}
	else
	{
		[dataTableView scrollRowToVisible:[dataTableView selectedRow]];
	}
}

//
// updateHistory
//
// Refresh and rescroll the history table.
//
- (void)updateHistory
{
	[historyTableView reloadData];
	[historyTableView scrollRowToVisible:[[dataManager history] count] - 1];
}

@end
