// ##############################################################
// ExpressionSymbols.m
// Magic Number Machine
//
// Created by Matt Gallagher on Sun May 04 2003
// Copyright (c) 2003 Matt Gallagher. All rights reserved.
// ##############################################################

#import "ExpressionSymbols.h"

//
// About ExpressionSymbols
//
// A single instance class that maintains the bezier paths for most drawable symbols
//
@implementation ExpressionSymbols

static NSMutableArray *symbols = nil;

//
// initialize
//
// Called once at program startup. Creates all the symbols.
//
+ (void)initialize
{
	NSArray				*stringSymbols;
	NSLayoutManager	*layoutManager;
	NSTextStorage		*text;
	NSBezierPath			*path;
	NSGlyph				*glyphs;
	int						i;
	int						j;
	int						numGlyphs;
	
	if (symbols != nil)
		return;
	
	// Create the array of strings that we plan to store in the Mutable array
	stringSymbols = [NSArray arrayWithObjects:
		[NSString stringWithUTF8String:"+"],		// 0
		[NSString stringWithUTF8String:"-"],		// 1
		[NSString stringWithUTF8String:"x"],		// 2
		[NSString stringWithUTF8String:"="],		// 3
		[NSString stringWithUTF8String:"sin"],		// 4	
		[NSString stringWithUTF8String:"cos"],		// 5
		[NSString stringWithUTF8String:"tan"],		// 6
		[NSString stringWithUTF8String:"h"],		// 7
		[NSString stringWithUTF8String:"Re"],		// 8
		[NSString stringWithUTF8String:"Im"],		// 9
		[NSString stringWithUTF8String:"abs"],		// 10
		[NSString stringWithUTF8String:"arg"],		// 11
		[NSString stringWithUTF8String:"and"],		// 12
		[NSString stringWithUTF8String:"or"],		// 13
		[NSString stringWithUTF8String:"xor"],		// 14
		[NSString stringWithUTF8String:"not"],		// 15
		[NSString stringWithUTF8String:"Rnd"],		// 16
		[NSString stringWithUTF8String:"log"],		// 17
		[NSString stringWithUTF8String:"ln"],		// 18
		[NSString stringWithUTF8String:"√"],		// 19
		[NSString stringWithUTF8String:"∑"],		// 20
		[NSString stringWithUTF8String:"10"],		// 21
		[NSString stringWithUTF8String:"e"],		// 22
		[NSString stringWithUTF8String:"!"],		// 23
		[NSString stringWithUTF8String:"i"],		// 24
		[NSString stringWithUTF8String:"π"],		// 25
		[NSString stringWithUTF8String:"%"],		// 26
		[NSString stringWithUTF8String:"nPr"],		// 27
		[NSString stringWithUTF8String:"nCr"],		// 28
		[NSString stringWithUTF8String:"("],		// 29
		[NSString stringWithUTF8String:")"],		// 30
		[NSString stringWithUTF8String:"•"],		// 31
		[NSString stringWithUTF8String:"2"],		// 32
		[NSString stringWithUTF8String:"-1"],		// 33
		nil
	];
	
	// Create the array for holding the symbols
	symbols = [[NSMutableArray arrayWithCapacity:[stringSymbols count]] retain];
	
	// Use a layout manager to get the glyphs for the string
	layoutManager = [[[NSLayoutManager alloc] init] autorelease];

	// Create a text storage area for the string
	text = [[NSTextStorage alloc] initWithString:[NSString stringWithUTF8String:""]];
	[text addLayoutManager:layoutManager];

	// Get the glyph/path representation of all the strings and store them appropriately
	for (i = 0; i < [stringSymbols count]; i++)
	{
		// Create a bezier path to contain the display
		path = [NSBezierPath bezierPath];
		
		if (i >= [stringSymbols count] - 2)
		{
			[text setAttributedString:
				[[[NSAttributedString alloc]
					initWithString:[stringSymbols objectAtIndex:i]
					attributes:[NSDictionary dictionaryWithObjectsAndKeys:[NSFont labelFontOfSize:16], NSFontAttributeName, nil]
				] autorelease]
			];
			[path moveToPoint:NSMakePoint(0, 12)];
		}
		else
		{
			[text setAttributedString:
				[[[NSAttributedString alloc]
					initWithString:[stringSymbols objectAtIndex:i]
					attributes:[NSDictionary dictionaryWithObjectsAndKeys:[NSFont labelFontOfSize:24], NSFontAttributeName, nil]
				] autorelease]
			];
			[path moveToPoint:NSMakePoint(0, 0)];
		}
		numGlyphs = [layoutManager numberOfGlyphs];
		glyphs = (NSGlyph *)malloc(sizeof(NSGlyph) * numGlyphs);
		for (j = 0; j < numGlyphs; j++)
		{
			glyphs[j] = [layoutManager glyphAtIndex:j];
		}
		[path
			appendBezierPathWithGlyphs:glyphs
			count:[text length]
			inFont:[text attribute:NSFontAttributeName atIndex:0 effectiveRange:NULL]
		];
		free(glyphs);
		
		[symbols addObject:path];
	}
	
	// Free the text storage area
	[text release];
}

//
// cleanup
//
// Called once at program shutdown to delete all the symbol paths.
//
+ (void)cleanup
{
	[symbols release];
}

//
// plusPath
//
// Returns the relevant bezier path.
//
+ (NSBezierPath *)plusPath
{
	NSBezierPath		*copy;
	
	copy = [NSBezierPath bezierPath];
	[copy appendBezierPath:[symbols objectAtIndex:0]];
	
	return copy;
}

//
// minusPath
//
// Returns the relevant bezier path.
//
+ (NSBezierPath *)minusPath
{
	NSBezierPath		*copy;
	
	copy = [NSBezierPath bezierPath];
	[copy appendBezierPath:[symbols objectAtIndex:1]];
	
	return copy;
}

//
// multiplyPath
//
// Returns the relevant bezier path.
//
+ (NSBezierPath *)multiplyPath
{
	NSBezierPath		*copy;
	
	copy = [NSBezierPath bezierPath];
	[copy appendBezierPath:[symbols objectAtIndex:2]];
	
	return copy;
}

//
// equalsPath
//
// Returns the relevant bezier path.
//
+ (NSBezierPath *)equalsPath
{
	NSBezierPath		*copy;
	
	copy = [NSBezierPath bezierPath];
	[copy appendBezierPath:[symbols objectAtIndex:3]];
	
	return copy;
}

//
// sinPath
//
// Returns the relevant bezier path.
//
+ (NSBezierPath *)sinPath
{
	NSBezierPath		*copy;
	
	copy = [NSBezierPath bezierPath];
	[copy appendBezierPath:[symbols objectAtIndex:4]];
	
	return copy;
}

//
// cosPath
//
// Returns the relevant bezier path.
//
+ (NSBezierPath *)cosPath
{
	NSBezierPath		*copy;
	
	copy = [NSBezierPath bezierPath];
	[copy appendBezierPath:[symbols objectAtIndex:5]];
	
	return copy;
}

//
// tanPath
//
// Returns the relevant bezier path.
//
+ (NSBezierPath *)tanPath
{
	NSBezierPath		*copy;
	
	copy = [NSBezierPath bezierPath];
	[copy appendBezierPath:[symbols objectAtIndex:6]];
	
	return copy;
}

//
// hypPath
//
// Returns the relevant bezier path.
//
+ (NSBezierPath *)hypPath
{
	NSBezierPath		*copy;
	
	copy = [NSBezierPath bezierPath];
	[copy appendBezierPath:[symbols objectAtIndex:7]];
	
	return copy;
}

//
// rePath
//
// Returns the relevant bezier path.
//
+ (NSBezierPath *)rePath
{
	NSBezierPath		*copy;
	
	copy = [NSBezierPath bezierPath];
	[copy appendBezierPath:[symbols objectAtIndex:8]];
	
	return copy;
}

//
// imPath
//
// Returns the relevant bezier path.
//
+ (NSBezierPath *)imPath
{
	NSBezierPath		*copy;
	
	copy = [NSBezierPath bezierPath];
	[copy appendBezierPath:[symbols objectAtIndex:9]];
	
	return copy;
}

//
// absPath
//
// Returns the relevant bezier path.
//
+ (NSBezierPath *)absPath
{
	NSBezierPath		*copy;
	
	copy = [NSBezierPath bezierPath];
	[copy appendBezierPath:[symbols objectAtIndex:10]];
	
	return copy;
}

//
// argPath
//
// Returns the relevant bezier path.
//
+ (NSBezierPath *)argPath
{
	NSBezierPath		*copy;
	
	copy = [NSBezierPath bezierPath];
	[copy appendBezierPath:[symbols objectAtIndex:11]];
	
	return copy;
}

//
// andPath
//
// Returns the relevant bezier path.
//
+ (NSBezierPath *)andPath
{
	NSBezierPath		*copy;
	
	copy = [NSBezierPath bezierPath];
	[copy appendBezierPath:[symbols objectAtIndex:12]];
	
	return copy;
}

//
// orPath
//
// Returns the relevant bezier path.
//
+ (NSBezierPath *)orPath
{
	NSBezierPath		*copy;
	
	copy = [NSBezierPath bezierPath];
	[copy appendBezierPath:[symbols objectAtIndex:13]];
	
	return copy;
}

//
// xorPath
//
// Returns the relevant bezier path.
//
+ (NSBezierPath *)xorPath
{
	NSBezierPath		*copy;
	
	copy = [NSBezierPath bezierPath];
	[copy appendBezierPath:[symbols objectAtIndex:14]];
	
	return copy;
}

//
// notPath
//
// Returns the relevant bezier path.
//
+ (NSBezierPath *)notPath
{
	NSBezierPath		*copy;
	
	copy = [NSBezierPath bezierPath];
	[copy appendBezierPath:[symbols objectAtIndex:15]];
	
	return copy;
}

//
// rndPath
//
// Returns the relevant bezier path.
//
+ (NSBezierPath *)rndPath
{
	NSBezierPath		*copy;
	
	copy = [NSBezierPath bezierPath];
	[copy appendBezierPath:[symbols objectAtIndex:16]];
	
	return copy;
}

//
// logPath
//
// Returns the relevant bezier path.
//
+ (NSBezierPath *)logPath
{
	NSBezierPath		*copy;
	
	copy = [NSBezierPath bezierPath];
	[copy appendBezierPath:[symbols objectAtIndex:17]];
	
	return copy;
}

//
// lnPath
//
// Returns the relevant bezier path.
//
+ (NSBezierPath *)lnPath
{
	NSBezierPath		*copy;
	
	copy = [NSBezierPath bezierPath];
	[copy appendBezierPath:[symbols objectAtIndex:18]];
	
	return copy;
}

//
// sqrtPath
//
// Returns the relevant bezier path.
//
+ (NSBezierPath *)sqrtPath
{
	NSBezierPath		*copy;
	
	copy = [NSBezierPath bezierPath];
	[copy appendBezierPath:[symbols objectAtIndex:19]];
	
	return copy;
}

//
// sigmaPath
//
// Returns the relevant bezier path.
//
+ (NSBezierPath *)sigmaPath
{
	NSBezierPath		*copy;
	
	copy = [NSBezierPath bezierPath];
	[copy appendBezierPath:[symbols objectAtIndex:20]];
	
	return copy;
}

//
// tenPath
//
// Returns the relevant bezier path.
//
+ (NSBezierPath *)tenPath
{
	NSBezierPath		*copy;
	
	copy = [NSBezierPath bezierPath];
	[copy appendBezierPath:[symbols objectAtIndex:21]];
	
	return copy;
}

//
// ePath
//
// Returns the relevant bezier path.
//
+ (NSBezierPath *)ePath
{
	NSBezierPath		*copy;
	
	copy = [NSBezierPath bezierPath];
	[copy appendBezierPath:[symbols objectAtIndex:22]];
	
	return copy;
}

//
// factorialPath
//
// Returns the relevant bezier path.
//
+ (NSBezierPath *)factorialPath
{
	NSBezierPath		*copy;
	
	copy = [NSBezierPath bezierPath];
	[copy appendBezierPath:[symbols objectAtIndex:23]];
	
	return copy;
}

//
// iPath
//
// Returns the relevant bezier path.
//
+ (NSBezierPath *)iPath
{
	NSBezierPath		*copy;
	
	copy = [NSBezierPath bezierPath];
	[copy appendBezierPath:[symbols objectAtIndex:24]];
	
	return copy;
}

//
// piPath
//
// Returns the relevant bezier path.
//
+ (NSBezierPath *)piPath
{
	NSBezierPath		*copy;
	
	copy = [NSBezierPath bezierPath];
	[copy appendBezierPath:[symbols objectAtIndex:25]];
	
	return copy;
}

//
// modPath
//
// Returns the relevant bezier path.
//
+ (NSBezierPath *)modPath
{
	NSBezierPath		*copy;
	
	copy = [NSBezierPath bezierPath];
	[copy appendBezierPath:[symbols objectAtIndex:26]];
	
	return copy;
}

//
// nprPath
//
// Returns the relevant bezier path.
//
+ (NSBezierPath *)nprPath
{
	NSBezierPath		*copy;
	
	copy = [NSBezierPath bezierPath];
	[copy appendBezierPath:[symbols objectAtIndex:27]];
	
	return copy;
}

//
// ncrPath
//
// Returns the relevant bezier path.
//
+ (NSBezierPath *)ncrPath
{
	NSBezierPath		*copy;
	
	copy = [NSBezierPath bezierPath];
	[copy appendBezierPath:[symbols objectAtIndex:28]];
	
	return copy;
}

//
// leftBracketPath
//
// Returns the relevant bezier path.
//
+ (NSBezierPath *)leftBracketPath
{
	NSBezierPath		*copy;
	
	copy = [NSBezierPath bezierPath];
	[copy appendBezierPath:[symbols objectAtIndex:29]];
	
	return copy;
}

//
// rightBracketPath
//
// Returns the relevant bezier path.
//
+ (NSBezierPath *)rightBracketPath
{
	NSBezierPath		*copy;
	
	copy = [NSBezierPath bezierPath];
	[copy appendBezierPath:[symbols objectAtIndex:30]];
	
	return copy;
}

//
// dotPath
//
// Returns the relevant bezier path.
//
+ (NSBezierPath *)dotPath
{
	NSBezierPath		*copy;
	
	copy = [NSBezierPath bezierPath];
	[copy appendBezierPath:[symbols objectAtIndex:31]];
	
	return copy;
}

//
// squarePath
//
// Returns the relevant bezier path.
//
+ (NSBezierPath *)squarePath
{
	NSBezierPath		*copy;
	
	copy = [NSBezierPath bezierPath];
	[copy appendBezierPath:[symbols objectAtIndex:32]];
	
	return copy;
}

//
// inversePath
//
// Returns the relevant bezier path.
//
+ (NSBezierPath *)inversePath
{
	NSBezierPath		*copy;
	
	copy = [NSBezierPath bezierPath];
	[copy appendBezierPath:[symbols objectAtIndex:33]];
	
	return copy;
}

@end
