// ##############################################################
//  MainWindow.m
//  Magic Number Machine
//
//  Created by Matt Gallagher on Sun Apr 20 2003.
//  Copyright (c) 2003 Matt Gallagher. All rights reserved.
// ##############################################################

#import "MainWindow.h"
#import "DataManager.h"

//
// About the MainWindow
//
// Only exists to catch shift and option keys, otherwise we use standard window
// behaviour.
//
@implementation MainWindow

//
// flagsChanged
//
// Tells the data manager when the state of the shift or option keys changes.
//
- (void)flagsChanged:(NSEvent*)theEvent
{
	unsigned int newFlags = [theEvent modifierFlags];
	
	[dataManager optionIsPressed:(newFlags & NSAlternateKeyMask) != 0];
	[dataManager shiftIsPressed:(newFlags & NSShiftKeyMask) != 0];

	[super flagsChanged:theEvent];
}

- (BOOL)performKeyEquivalent:(NSEvent *)theEvent
{
	NSEvent *alteredEvent;
	unsigned int modifiers = [theEvent modifierFlags];
	NSString *chars = [theEvent characters];
	NSString *charsWithout = [[theEvent charactersIgnoringModifiers] lowercaseString];
	BOOL needNewEvent = NO;
	
	if ([theEvent type] == NSKeyUp)
	{
		return false;
	}
	
	// Handle the window close shortcut (which is not associated with any UI
	// element).
	if ((modifiers & NSCommandKeyMask) &&
		!(modifiers & (NSShiftKeyMask | NSAlternateKeyMask | NSControlKeyMask)) &&
		[charsWithout isEqualTo:@"w"])
	{
		[self performClose:self];
		return YES;
	}
	
	// Handle the shortcut for the AC button ("clear" key) manually. This is
	// because "clear" and "ESC" are treated as synonyms by the standard
	// performKeyEquivalent processing and we wish to differentiate.
	if ([charsWithout isEqualTo:[NSString stringWithUTF8String:""]])
	{
		const int acButtonTag = 1234;
		
		[[[self contentView] viewWithTag:acButtonTag] performClick:self];
		return YES;
	}
	
	// Strip the shift key and alternate key so that Shft and Opt buttons
	// work in conjunction with other buttons.
	if (modifiers & (NSShiftKeyMask | NSAlternateKeyMask))
	{
		modifiers &= ~(NSShiftKeyMask | NSAlternateKeyMask);
		needNewEvent = YES;
	}
	
	// Convert the locale radix point to a period
	if ([charsWithout isEqualTo:[[NSLocale currentLocale] objectForKey:NSLocaleDecimalSeparator]])
	{
		chars = [NSString stringWithCString:"."];
		charsWithout = chars;
		needNewEvent = YES;
	}
	
	// Convert equals and return to be a synonym of "return"
	else if ([charsWithout isEqualTo:@"="])
	{
		chars = [NSString stringWithCString:"\r"];
		charsWithout = chars;
		needNewEvent = YES;
	}
	
	// Let "PageUp" key make the window floatin
	else if ([theEvent keyCode] == 116 && [theEvent type] == NSKeyDown)
	{
		[self setLevel:NSFloatingWindowLevel];
		needNewEvent = YES;
	}

	// Let "PageDown" key make the window normal again
	else if ([theEvent keyCode] == 121 && [theEvent type] == NSKeyDown)
	{
		[self setLevel:NSNormalWindowLevel];
		needNewEvent = YES;
	}
	
	if (needNewEvent)
	{
		// Dispatch the event with shift and option stripped out
		alteredEvent = [NSEvent
			keyEventWithType:[theEvent type]
			location:[theEvent locationInWindow]
			modifierFlags:modifiers
			timestamp:[theEvent timestamp]
			windowNumber:[theEvent windowNumber]
			context:[theEvent context]
			characters:chars
			charactersIgnoringModifiers:charsWithout
			isARepeat:[theEvent isARepeat]
			keyCode:[theEvent keyCode]
		];
	}
	else
	{
		alteredEvent = theEvent;
	}
	
	return [super performKeyEquivalent:alteredEvent];
}

@end
